let balance = window.initBalance;
let wager = window.initWager;
let currentLane = 0;
let playing = false;
let lanes = 8;
let laneItems = [];
let currentMult = 1.00;
let gameOver = false;

const chickenImg = '🐔';
const fixedMultipliers = [1.2, 1.5, 2.0, 3.0, 5.0, 8.0, 12.0, 20.0];

function renderRoad() {
    const roadDiv = document.getElementById('road');
    roadDiv.innerHTML = '';
    // Remove crash indicator and fire-effect rendering
    for (let i = 0; i < lanes; i++) {
        let laneDiv = document.createElement('div');
        laneDiv.className = 'lane';
        if (playing && i < currentLane) {
            laneDiv.classList.add('completed');
        }
        if ((playing && i === currentLane && !gameOver) || (!playing && i === 0)) {
            laneDiv.classList.add('active');
        }
        // Show chicken at starting position (lane 0) when game hasn't started
        if (!playing && i === 0) {
            let chickenDiv = document.createElement('div');
            chickenDiv.className = 'chicken';
            chickenDiv.innerHTML = chickenImg;
            laneDiv.appendChild(chickenDiv);
        }
        // Show chicken at current position during gameplay
        else if (playing && i === currentLane && !gameOver) {
            let chickenDiv = document.createElement('div');
            chickenDiv.className = 'chicken chicken-jump';
            chickenDiv.innerHTML = chickenImg;
            chickenDiv.addEventListener('animationend', function() {
                chickenDiv.classList.remove('chicken-jump');
            }, { once: true });
            laneDiv.appendChild(chickenDiv);
            let multDiv = document.createElement('div');
            multDiv.className = 'item multiplier animate-multiplier';
            multDiv.innerHTML = fixedMultipliers[i].toFixed(2) + 'x';
            laneDiv.appendChild(multDiv);
        } else {
            let multDiv = document.createElement('div');
            multDiv.className = 'item multiplier';
            multDiv.innerHTML = fixedMultipliers[i].toFixed(2) + 'x';
            laneDiv.appendChild(multDiv);
        }
        roadDiv.appendChild(laneDiv);
    }
    // Smoothly slide road left on chicken jump (mobile)
    if (window.innerWidth <= 600) {
        const activeLane = roadDiv.querySelector('.lane.active');
        if (activeLane) {
            const targetScroll = activeLane.offsetLeft;
            const startScroll = roadDiv.scrollLeft;
            const distance = targetScroll - startScroll;
            const duration = 400; // ms
            let startTime = null;
            function animateScroll(timestamp) {
                if (!startTime) startTime = timestamp;
                const elapsed = timestamp - startTime;
                const progress = Math.min(elapsed / duration, 1);
                roadDiv.scrollLeft = startScroll + distance * progress;
                if (progress < 1) {
                    requestAnimationFrame(animateScroll);
                }
            }
            requestAnimationFrame(animateScroll);
        }
    }
    // No crash indicator or fire-effect
    // updateCrashIndicator();
}

function updateInfoBar() {
    let currMult = playing ? (fixedMultipliers[currentLane] || 1.00) : 1.00;
    document.getElementById('currentMult') && (document.getElementById('currentMult').textContent = currMult.toFixed(2) + 'x');
    document.getElementById('potentialAmt') && (document.getElementById('potentialAmt').textContent = '₹' + (wager * currMult).toFixed(2));
    updateCashoutAmt();
}

function updateCrashIndicator() {
    const crashIndicator = document.getElementById('crashIndicator');
    if (crashIndicator) {
        // Calculate position based on current lane
        // The indicator should move from right to left as the chicken progresses
        const roadWidth = document.getElementById('road').offsetWidth;
        const laneWidth = roadWidth / lanes;
        const currentPosition = (lanes - 1 - currentLane) * laneWidth;
        
        // Add some offset to center the indicator in the lane
        const offset = laneWidth / 2 - 4; // 4 is half the indicator width
        const finalPosition = Math.max(0, currentPosition + offset);
        
        crashIndicator.style.transform = `translateX(${finalPosition}px)`;
    }
}

function updateBalance() {
    document.getElementById('balance').textContent = balance.toFixed(2);
    var bh = document.getElementById('balanceHeader');
    if (bh) bh.textContent = balance.toFixed(2);
    checkLowBalance();
}

// Function to refresh balance from server
function refreshBalance() {
    fetch('get_balance.php')
        .then(response => response.json())
        .then(data => {
            if (data.balance !== undefined) {
                balance = parseFloat(data.balance);
                updateBalance();
            }
        })
        .catch(error => console.error('Error fetching balance:', error));
}

function setWager(amount) {
    if (playing) return;
    wager = amount;
    document.getElementById('wager').textContent = wager;
    updateInfoBar();
}

function changeWager(delta) {
    if (playing) return;
    wager += delta;
    if (wager < 1) wager = 1;
    document.getElementById('wager').textContent = wager;
    updateInfoBar();
}

function setDifficulty(diff) {
    // No effect in this version
}

function showPlayButton() {
    document.getElementById('playBtn').style.display = '';
    document.getElementById('goBtn').style.display = 'none';
    document.getElementById('cashoutBtn').style.display = 'none';
}

function showGoAndCashoutButtons() {
    document.getElementById('playBtn').style.display = 'none';
    document.getElementById('goBtn').style.display = '';
    document.getElementById('cashoutBtn').style.display = '';
}

function updateCashoutAmt() {
    let currMult = playing ? (fixedMultipliers[currentLane] || 1.00) : 1.00;
    document.getElementById('cashoutAmt').textContent = (wager * currMult).toFixed(2);
}

function startGame() {
    if (balance < 1) {
        checkLowBalance();
        return;
    }
    if (wager > balance) {
        checkLowBalance();
        return;
    }
    balance -= wager;
    updateBalance();
    currentLane = 0;
    currentMult = fixedMultipliers[0];
    playing = true;
    gameOver = false;
    showGoAndCashoutButtons();
    document.getElementById('gameMessage').textContent = '';
    renderRoad();
    updateInfoBar();
    updateCrashIndicator();
}

function goNext() {
    if (!playing || gameOver) return;
    // Get crash level from admin settings
    fetch('backend/get_crash_level.php')
        .then(response => response.json())
        .then(data => {
            const crashLevel = data.crash_level || 5;
            if (currentLane < lanes - 1) {
                currentLane++;
                currentMult = fixedMultipliers[currentLane];
                renderRoad();
                updateInfoBar();
                updateCrashIndicator();
                // Check if chicken should crash at the exact crash level
                if (currentLane + 1 >= crashLevel) {
                    // Chicken crashes at the specified level!
                    gameOver = true;
                    playing = false;
                    showPlayButton();
                    if (typeof window.playLossSound === 'function') window.playLossSound();
                    showCrashPopup(crashLevel, wager);
                }
            } else {
                // Max level reached, auto-cashout
                cashout();
            }
        })
        .catch(error => {
            console.error('Error fetching crash level:', error);
            if (currentLane < lanes - 1) {
                currentLane++;
                currentMult = fixedMultipliers[currentLane];
                renderRoad();
                updateInfoBar();
                updateCrashIndicator();
            } else {
                cashout();
            }
        });
}

function cashout() {
    if (!playing) return;
    let winAmt = wager * (fixedMultipliers[currentLane] || 1.00);
    let profit = winAmt - wager; // Only profit should be added to wallet
    balance += winAmt;
    updateBalance();

    // Backend update with debug logging (only profit)
    fetch('update_balance.php', {
        method: 'POST',
        headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
        body: 'amount=' + encodeURIComponent(profit)
    })
    .then(res => res.json())
    .then(data => console.log('Balance update response:', data));

    showPlayButton();
    playing = false;
    gameOver = false;
    if (typeof window.playWinSound === 'function') window.playWinSound();
    showCashoutPopup(currentLane + 1, profit);
}

function resetGame() {
    playing = false;
    currentLane = 0;
    currentMult = 1.00;
    gameOver = false;
    showPlayButton();
    renderRoad();
    updateInfoBar();
    updateCrashIndicator();
    setTimeout(() => {
        document.getElementById('gameMessage').textContent = '';
    }, 1000);
}

// Initial render
renderRoad();
updateInfoBar();
updateBalance();

// Sidebar toggles (UI only)
document.addEventListener('DOMContentLoaded', function() {
    const soundToggle = document.getElementById('soundToggle');
    const musicToggle = document.getElementById('musicToggle');
    const spaceToggle = document.getElementById('spaceToggle');
    const bgMusic = document.getElementById('bgMusic');
    const goSound = document.getElementById('goSound');
    const winSound = document.getElementById('winSound');
    const lossSound = document.getElementById('lossSound');

    // Music toggle logic
    function updateMusic() {
        if (musicToggle.checked) {
            bgMusic.volume = 0.5;
            bgMusic.play().catch(()=>{});
        } else {
            bgMusic.pause();
        }
    }
    musicToggle.onchange = updateMusic;
    // Start music if toggle is ON by default
    if (musicToggle.checked) {
        setTimeout(updateMusic, 500);
    }

    // GO sound logic
    window.playGoSound = function() {
        if (soundToggle.checked) {
            goSound.currentTime = 0;
            goSound.volume = 0.7;
            goSound.play().catch(()=>{});
        }
    };
    
    // Win sound logic
    window.playWinSound = function() {
        if (soundToggle.checked) {
            winSound.currentTime = 0;
            winSound.volume = 0.8;
            winSound.play().catch(()=>{});
        }
    };
    
    // Loss sound logic
    window.playLossSound = function() {
        if (soundToggle.checked) {
            lossSound.currentTime = 0;
            lossSound.volume = 0.8;
            lossSound.play().catch(()=>{});
        }
    };
    
    // Default: sound ON
    soundToggle.checked = true;

    // Set wager to 1 or balance (whichever is lower) on page load
    if (balance < 1) {
        wager = balance;
    } else {
        wager = 1;
    }
    document.getElementById('wager').textContent = wager;
    updateInfoBar();
    // Show/hide low balance message
    checkLowBalance();
    showPlayButton();
});

function checkLowBalance() {
    const lowMsg = document.getElementById('lowBalanceMsg');
    const playBtn = document.getElementById('goBtn');
    if (balance < 1) {
        if (lowMsg) lowMsg.style.display = '';
        if (playBtn) playBtn.disabled = true;
    } else {
        if (lowMsg) lowMsg.style.display = 'none';
        if (playBtn) playBtn.disabled = false;
    }
}

// Popup functions
function showCrashPopup(level, lostAmount) {
    document.getElementById('crashLevel').textContent = level;
    document.getElementById('lostAmount').textContent = '₹' + lostAmount.toFixed(2);
    document.getElementById('crashPopup').style.display = 'flex';

    // Backend update for loss
    fetch('update_balance.php', {
        method: 'POST',
        headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
        body: 'amount=' + encodeURIComponent(-wager)
    })
    .then(res => res.json())
    .then(data => console.log('Balance update response (loss):', data));
}

function showCashoutPopup(level, wonAmount) {
    document.getElementById('cashoutLevel').textContent = level;
    document.getElementById('wonAmount').textContent = '₹' + wonAmount.toFixed(2);
    document.getElementById('cashoutPopup').style.display = 'flex';
    
    // Win sound is already played in cashout() function
}

function closeCrashPopup() {
    document.getElementById('crashPopup').style.display = 'none';
    resetGame();
}

function closeCashoutPopup() {
    document.getElementById('cashoutPopup').style.display = 'none';
    resetGame();
}

// Patch goNext to play sound
const origGoNext = window.goNext;
window.goNext = function() {
    if (typeof window.playGoSound === 'function') window.playGoSound();
    origGoNext();
}; 